#!/bin/sh
### BEGIN INIT INFO
# Provides:             volatile
# Required-Start:       $local_fs
# Required-Stop:      $local_fs
# Default-Start:        S
# Default-Stop:
# Short-Description:  Populate the volatile filesystem
### END INIT INFO

# Get ROOT_DIR
DIRNAME=`dirname $0`
CMD=$1
ROOT_DIR=`echo $DIRNAME | sed -ne 's:/etc/.*::p'`

[ -e ${ROOT_DIR}/etc/default/rcS ] && . ${ROOT_DIR}/etc/default/rcS
[ -e /etc/exorint.funcs ] && . /etc/exorint.funcs

# When running populate-volatile.sh at rootfs time, disable cache.
[ -n "$ROOT_DIR" ] && VOLATILE_ENABLE_CACHE=no

CFGDIR="${ROOT_DIR}/etc/default/volatiles"
TMPROOT="${ROOT_DIR}/var/volatile/tmp"
COREDEF="00_core"
SYSLOG="/etc/init.d/syslog"
MOUNT="/bin/mount"

VOLATILELOGDIR="/var/volatile/log"
PERSISTENTLOGDIR="$DATAMNT/log"

AUDITSIMLINK="$VOLATILELOGDIR/audit"
PERSISTENTAUDITDIR="$DATAMNT/audit"
AUDITFALLOCATEFILE="$DATAMNT/audit.fallocate"
AUDITPARTITIONPERCENT=90

die() {
	echo "$@" >&2
	[ "$CMD" != "start" ] && exit 1
}

create_file() {
	EXEC="
	touch \"$1\";
	chown ${TUSER}.${TGROUP} $1 || echo \"Failed to set owner -${TUSER}- for -$1-.\" >/dev/tty0 2>&1;
	chmod ${TMODE} $1 || echo \"Failed to set mode -${TMODE}- for -$1-.\" >/dev/tty0 2>&1 "

	test "$VOLATILE_ENABLE_CACHE" = yes && echo "$EXEC" >> /etc/volatile.cache.build

	[ -e "$1" ] && {
		[ "${VERBOSE}" != "no" ] && echo "Target already exists. Skipping."
	} || {
		if [ -z "$ROOT_DIR" ]; then
			eval $EXEC &
		else
			# Creating some files at rootfs time may fail and should fail,
			# but these failures should not be logged to make sure the do_rootfs
			# process doesn't fail. This does no harm, as this script will
			# run on target to set up the correct files and directories.
			eval $EXEC > /dev/null 2>&1
		fi
	}
}

mk_dir() {
	EXEC="
	mkdir -p \"$1\";
	chown ${TUSER}.${TGROUP} $1 || echo \"Failed to set owner -${TUSER}- for -$1-.\" >/dev/tty0 2>&1;
	chmod ${TMODE} $1 || echo \"Failed to set mode -${TMODE}- for -$1-.\" >/dev/tty0 2>&1 "

	test "$VOLATILE_ENABLE_CACHE" = yes && echo "$EXEC" >> /etc/volatile.cache.build
	[ -e "$1" ] && {
		[ "${VERBOSE}" != "no" ] && echo "Target already exists. Skipping."
	} || {
		if [ -z "$ROOT_DIR" ]; then
			eval $EXEC
		else
			# For the same reason with create_file(), failures should
			# not be logged.
			eval $EXEC > /dev/null 2>&1
		fi
	}
}

link_file() {
	EXEC="
	if [ -L \"$2\" ]; then
		if [ \"\$(readlink -f \"$2\")\" != \"\$(readlink -f \"$1\")\" ]; then
			rm -f \"$2\";
			[ \$? = 0 ] && ln -sf \"$1\" \"$2\";
			[ ! \$? = 0 ] && target=\$(readlink -f \"$1\") && mount -o bind \$target \"$2\";
		fi
		elif [ -d \"$2\" ]; then
		cp -a $2/* $1 2>/dev/null;
		cp -a $2/.[!.]* $1 2>/dev/null;
		rm -rf \"$2\";
		[ \$? = 0 ] && ln -sf \"$1\" \"$2\";
		[ ! \$? = 0 ] && target=\$(readlink -f \"$1\") && mount -o bind \$target \"$2\";
	else
		ln -sf \"$1\" \"$2\";
		[ ! \$? = 0 ] && target=\$(readlink -f \"$1\") && mount -o bind \$target \"$2\";
	fi
        "

	test "$VOLATILE_ENABLE_CACHE" = yes && echo "	$EXEC" >> /etc/volatile.cache.build

	if [ -z "$ROOT_DIR" ]; then
		eval $EXEC &
	else
		# For the same reason with create_file(), failures should
		# not be logged.
		eval $EXEC > /dev/null 2>&1
	fi
}

check_requirements() {
	cleanup() {
		rm "${TMP_INTERMED}"
		rm "${TMP_DEFINED}"
		rm "${TMP_COMBINED}"
	}

	CFGFILE="$1"
	[ `basename "${CFGFILE}"` = "${COREDEF}" ] && return 0

	TMP_INTERMED="${TMPROOT}/tmp.$$"
	TMP_DEFINED="${TMPROOT}/tmpdefined.$$"
	TMP_COMBINED="${TMPROOT}/tmpcombined.$$"

	cat ${ROOT_DIR}/etc/passwd | sed 's@\(^:\)*:.*@\1@' | sort | uniq > "${TMP_DEFINED}"
	cat ${CFGFILE} | grep -v "^#" | cut -s -d " " -f 2 > "${TMP_INTERMED}"
	cat "${TMP_DEFINED}" "${TMP_INTERMED}" | sort | uniq > "${TMP_COMBINED}"
	NR_DEFINED_USERS="`cat "${TMP_DEFINED}" | wc -l`"
	NR_COMBINED_USERS="`cat "${TMP_COMBINED}" | wc -l`"

	[ "${NR_DEFINED_USERS}" -ne "${NR_COMBINED_USERS}" ] && {
		echo "Undefined users:"
		diff "${TMP_DEFINED}" "${TMP_COMBINED}" | grep "^>"
		cleanup
		return 1
	}


	cat ${ROOT_DIR}/etc/group | sed 's@\(^:\)*:.*@\1@' | sort | uniq > "${TMP_DEFINED}"
	cat ${CFGFILE} | grep -v "^#" | cut -s -d " " -f 3 > "${TMP_INTERMED}"
	cat "${TMP_DEFINED}" "${TMP_INTERMED}" | sort | uniq > "${TMP_COMBINED}"

	NR_DEFINED_GROUPS="`cat "${TMP_DEFINED}" | wc -l`"
	NR_COMBINED_GROUPS="`cat "${TMP_COMBINED}" | wc -l`"

	[ "${NR_DEFINED_GROUPS}" -ne "${NR_COMBINED_GROUPS}" ] && {
		echo "Undefined groups:"
		diff "${TMP_DEFINED}" "${TMP_COMBINED}" | grep "^>"
		cleanup
		return 1
	}

	# Add checks for required directories here

	cleanup
	return 0
}

apply_cfgfile() {
	CFGFILE="$1"

	check_requirements "${CFGFILE}" || {
		echo "Skipping ${CFGFILE}"
		return 1
	}

	cat ${CFGFILE} | grep -v "^#" | \
		while read LINE; do
		eval `echo "$LINE" | sed -n "s/\(.*\)\ \(.*\) \(.*\)\ \(.*\)\ \(.*\)\ \(.*\)/TTYPE=\1 ; TUSER=\2; TGROUP=\3; TMODE=\4; TNAME=\5 TLTARGET=\6/p"`
		TNAME=${ROOT_DIR}${TNAME}
		[ "${VERBOSE}" != "no" ] && echo "Checking for -${TNAME}-."

		[ "${TTYPE}" = "l" ] && {
			TSOURCE="$TLTARGET"
			[ "${VERBOSE}" != "no" ] && echo "Creating link -${TNAME}- pointing to -${TSOURCE}-."
			link_file "${TSOURCE}" "${TNAME}"
			continue
		}

		[ -L "${TNAME}" ] && {
			[ "${VERBOSE}" != "no" ] && echo "Found link."
			NEWNAME=`ls -l "${TNAME}" | sed -e 's/^.*-> \(.*\)$/\1/'`
			echo ${NEWNAME} | grep -v "^/" >/dev/null && {
				TNAME="`echo ${TNAME} | sed -e 's@\(.*\)/.*@\1@'`/${NEWNAME}"
				[ "${VERBOSE}" != "no" ] && echo "Converted relative linktarget to absolute path -${TNAME}-."
			} || {
				TNAME="${NEWNAME}"
				[ "${VERBOSE}" != "no" ] && echo "Using absolute link target -${TNAME}-."
			}
		}

		case "${TTYPE}" in
			"f")  [ "${VERBOSE}" != "no" ] && echo "Creating file -${TNAME}-."
				create_file "${TNAME}" &
				;;
			"d")  [ "${VERBOSE}" != "no" ] && echo "Creating directory -${TNAME}-."
				mk_dir "${TNAME}"
				# Add check to see if there's an entry in fstab to mount.
				;;
			*)    [ "${VERBOSE}" != "no" ] && echo "Invalid type -${TTYPE}-."
				continue
				;;
		esac
	done
	return 0
}

logP=$( cat $CFGDIR/pLogFlag 2>/dev/null )


# Audit Management Procedure BEGIN ------
if [ -n "$CMD" -a "$CMD" != "update" ]; then
    AUDIT_CHANGED="false"
    AUDIT_SIZE_DESIRED=$(exorint_fparam 'options\logs\audit\size')
    # Set default in case EPAD is not running yet
    [ -z "$AUDIT_SIZE_DESIRED" ] && AUDIT_SIZE_DESIRED=16
    # Set audit always persistent
    mkdir -p $PERSISTENTAUDITDIR

    # Calculates the maximum size by estimating a lower percentage
    AUDIT_PARTITION_SIZE=$(df -m $DATAMNT | tail -n1 | awk '{print $4}')
    AUDIT_PARTITION_SIZE=$(($AUDIT_PARTITION_SIZE * $AUDITPARTITIONPERCENT / 100 ))

    if [ $AUDIT_SIZE_DESIRED -gt $AUDIT_PARTITION_SIZE ]; then
        die "Not enough space in $DATAMNT (request:$AUDIT_SIZE_DESIRED available:$AUDIT_PARTITION_SIZE)"
    fi
    # Audit fallocate already exist, get file size and umount+remove it
    if [ -f $AUDITFALLOCATEFILE ]; then
        AUDIT_SIZE_CURRENT=$(du -m "$AUDITFALLOCATEFILE" | cut -f1)
        if [ $AUDIT_SIZE_CURRENT != $AUDIT_SIZE_DESIRED ]; then
             /etc/init.d/syslog stop
             umount $AUDITFALLOCATEFILE
             rm $AUDITFALLOCATEFILE
        fi
    fi

    # Audit fallocate doesn't exist or size must changed, preallocate and create an ext4 fs on it
    if [ -z "$AUDIT_SIZE_CURRENT" ] || [ $AUDIT_SIZE_CURRENT != $AUDIT_SIZE_DESIRED ]; then
        fallocate -l ${AUDIT_SIZE_DESIRED}M $AUDITFALLOCATEFILE
        mke2fs -m 1 -t ext2 -L \"audit\" $AUDITFALLOCATEFILE || die "Failed mke2fs of $AUDITFALLOCATEFILE"
        AUDIT_CHANGED="true"
    fi

    # Check and mount fallocated audit and create symbolic link if needed
    if [ -f $AUDITFALLOCATEFILE ] && ! mountpoint -q $PERSISTENTAUDITDIR; then
        exorint_extfsck $AUDITFALLOCATEFILE
        mount $AUDITFALLOCATEFILE $PERSISTENTAUDITDIR || die "Failed mount of $PERSISTENTAUDITDIR"
    fi
    if [ ! -e $AUDITSIMLINK ]; then
        ln -s $PERSISTENTAUDITDIR $AUDITSIMLINK
    fi
    # Restart rsyslog so that it notices the changes (on-the-fly)
    if [ "$AUDIT_CHANGED" = "true" ]; then
        /etc/rsyslog-rotate.sh $AUDIT_SIZE_DESIRED > /etc/logrotate.d/logrotate.rsyslog
        sync
        $SYSLOG restart
    fi
fi
# Audit Management Procedure END --------

# Move log files to a persistent fs
if [ "$CMD" = "enable" ]; then
    # Check if it's already in persistence mode
    [ "$logP" = "y" ] && exit 0
    echo "y" > $CFGDIR/pLogFlag
    mkdir -p $PERSISTENTLOGDIR
    # Logrotate need a secure directory (NOT world writable or writable by group which is not "root")
    chmod 750 $PERSISTENTLOGDIR
    $SYSLOG stop
    cp -rpd $VOLATILELOGDIR/* $PERSISTENTLOGDIR
    rm -rf $VOLATILELOGDIR/*
    mount -o bind $PERSISTENTLOGDIR $VOLATILELOGDIR
    # Set messages total size 32 * 4 = 128MB
    /etc/rsyslog-rotate.sh $AUDIT_SIZE_DESIRED 32 > /etc/logrotate.d/logrotate.rsyslog
    sync
    $SYSLOG start
    exit 0
fi

# Restore default log files location
if [ "$CMD" = "disable" ]; then
    # Check if persistence mode is really enabled
    [ ! "$logP" = "y" ] && exit 0
    echo "n" > $CFGDIR/pLogFlag
    $SYSLOG	stop
    umount -l $VOLATILELOGDIR
    # Tmpfs size is limited in /var/volatile. Remove messages logs first to save space
    for mlog in $( find $PERSISTENTLOGDIR/messages* ); do
        rm $mlog
    done
    cp -rpd $PERSISTENTLOGDIR/* $VOLATILELOGDIR
    /etc/rsyslog-rotate.sh $AUDIT_SIZE_DESIRED > /etc/logrotate.d/logrotate.rsyslog
    sync
    $SYSLOG start
    rm -rf $PERSISTENTLOGDIR
    exit 0
fi

# At boot time, mount persistent log location if persistence is enabled
if [ "$logP" = "y" ]; then

    if ( mount | grep $DATAPARTITION | grep -q -v rw, ); then
        [ "$ENABLE_ROOTFS_FSCK" = "yes" ] && exorint_extfsck $DATAPARTITION
        mount -o remount,rw $DATAMNT
        mount -o remount,rw /home
    fi

    mkdir -p $PERSISTENTLOGDIR
    # Logrotate need a secure directory (NOT world writable or writable by group which is not "root")
    chmod 750 $PERSISTENTLOGDIR
    mount -o bind $PERSISTENTLOGDIR $VOLATILELOGDIR
fi

[ "${VERBOSE}" != "no" ] && echo "Populating volatile Filesystems."

clearcache=0
exec 9</proc/cmdline
while read line <&9
do
	case "$line" in
		*clearcache*)  clearcache=1
			       ;;
		*)	       continue
			       ;;
	esac
done
exec 9>&-

if test -e ${ROOT_DIR}/etc/volatile.cache.sh -a "$VOLATILE_ENABLE_CACHE" = "yes" -a "x$1" != "xupdate" -a "x$clearcache" = "x0"
then
	sh ${ROOT_DIR}/etc/volatile.cache.sh
else
	rm -f ${ROOT_DIR}/etc/volatile.cache.sh ${ROOT_DIR}/etc/volatile.cache.build
	for file in `ls -1 "${CFGDIR}" | sort`; do
		apply_cfgfile "${CFGDIR}/${file}"
	done

	[ -e ${ROOT_DIR}/etc/volatile.cache.build ] && sync && mv ${ROOT_DIR}/etc/volatile.cache.build ${ROOT_DIR}/etc/volatile.cache.sh
fi

if [ -z "${ROOT_DIR}" ] && [ -f /etc/ld.so.cache ] && [ ! -f /var/run/ld.so.cache ]
then
	ln -s /etc/ld.so.cache /var/run/ld.so.cache
fi

echo 'Populate-Volatile fully ended'
